#version 140
#extension GL_EXT_gpu_shader4 : enable

uniform sampler2D texture0;        // dancer feed
uniform vec2 textureSize0;         // resolution of dancer texture

uniform float depthAmount;         // max pixel shift in pixels (default 5.0)
uniform float maskSoftness;        // alpha edge softness (default 0.15)
uniform float gradientStrength;    // cylindrical gradient strength (default 0.8)
uniform float grayBlend;           // blend between cyl (0.0) and grayscale (1.0), default 0.5
uniform float eyeSign;             // +1.0 = right eye, -1.0 = left eye
uniform float debugMode;           // 0.0 = normal, 1.0 = blended mask, 2.0 = cyl-only mask

varying vec4 gl_TexCoord[];

// Scan horizontally from a starting UV until alpha < threshold or maxSteps reached
float scanEdge(vec2 uv, float dir, float maxSteps, float stepSize, float threshold) {
    for (float i = 0.0; i < maxSteps; i += 1.0) {
        float a = texture2D(texture0, uv + vec2(i * stepSize * dir, 0.0)).a;
        if (a < threshold) {
            return uv.x + i * stepSize * dir;
        }
    }
    return uv.x + dir * maxSteps * stepSize;
}

void main(void) {
    vec2 uv = gl_TexCoord[0].xy;

    // Defaults
    float depthAmt     = (depthAmount      != 0.0) ? depthAmount      : 5.0;
    float maskSoft     = (maskSoftness     != 0.0) ? maskSoftness     : 0.15;
    float gradStrength = (gradientStrength != 0.0) ? gradientStrength : 0.8;
    float grayMix      = (grayBlend        != 0.0) ? grayBlend        : 0.5;
    float eyeDir       = (eyeSign          != 0.0) ? eyeSign          : 1.0;

    // Sample center pixel
    vec4 rgbM  = texture2D(texture0, uv).rgba;
    float alphaCenter = rgbM.a;

    // Flood-fill mask from alpha
    float insideMask = smoothstep(0.0, maskSoft, alphaCenter);

    // --- Horizontal scan to find left/right edges ---
    float stepSize = 1.0 / textureSize0.x; // one pixel in UV space
    float maxSteps = 128.0;                // how far to scan in each direction
    float threshold = 0.01;                // alpha threshold for "transparent"

    float leftEdge  = scanEdge(uv, -1.0, maxSteps, stepSize, threshold);
    float rightEdge = scanEdge(uv,  1.0, maxSteps, stepSize, threshold);

    // Remap current pixel's X into 0–1 within the silhouette span
    float modelU = clamp((uv.x - leftEdge) / max(rightEdge - leftEdge, 0.0001), 0.0, 1.0);

    // Cylindrical gradient in silhouette space
    float cyl = clamp(1.0 - abs(modelU - 0.5) * 2.0, 0.0, 1.0);
    cyl = mix(1.0, cyl, gradStrength);

    // Grayscale luminance
    float gray = dot(rgbM.rgb, vec3(0.299, 0.587, 0.114));

    // Depth maps
    float blendedDepth = mix(cyl, gray, grayMix);
    float depthMask    = insideMask * blendedDepth;
    float cylMask      = insideMask * cyl;

    // Debug modes
    if (debugMode > 1.5) {
        gl_FragColor = vec4(vec3(cylMask), alphaCenter);
        return;
    }
    if (debugMode > 0.5) {
        gl_FragColor = vec4(vec3(depthMask), alphaCenter);
        return;
    }

    // Normal stereo
    float maxShift = depthAmt / textureSize0.x;
    vec2 offset = vec2(eyeDir * (depthMask - 0.5) * maxShift, 0.0);

    vec4 eyeView = texture2D(texture0, uv + offset).rgba;
    float outAlpha = alphaCenter * insideMask;
    eyeView.rgb *= insideMask;
    eyeView.a = outAlpha;

    gl_FragColor = eyeView;
}
